# 第22课：异常处理 try...except

## 什么是异常？

异常是程序运行时发生的错误。如果不处理异常，程序会崩溃并停止运行。使用 try...except 可以捕获并处理这些错误，让程序继续运行。

## 为什么要用异常处理？

### 1. 防止程序崩溃

**不使用异常处理：**
```python
age = int(input("请输入年龄："))  # 如果用户输入"abc"，程序直接崩溃
print(f"你的年龄是{age}")
print("程序继续运行...")  # 这行代码永远不会执行
```

**使用异常处理：**
```python
try:
    age = int(input("请输入年龄："))
    print(f"你的年龄是{age}")
except ValueError:
    print("输入错误，请输入数字")
    age = 0
print("程序继续运行...")  # 即使出错，这行代码也会执行
```

### 2. 提升用户体验

没有异常处理时，用户看到的是难以理解的错误信息：
```
Traceback (most recent call last):
  File "test.py", line 1, in <module>
    age = int(input("请输入年龄："))
ValueError: invalid literal for int() with base 10: 'abc'
```

使用异常处理后，用户看到的是友好的提示：
```
输入错误，请输入数字
```

### 3. 让程序更加健壮

在实际应用中，很多情况是不可控的：
- 用户可能输入错误的数据
- 文件可能不存在或被删除
- 网络连接可能中断
- 磁盘空间可能不足

使用异常处理可以让程序应对这些意外情况，而不是直接崩溃。

### 4. 便于调试和维护

```python
try:
    # 复杂的业务逻辑
    result = process_data(user_input)
    save_to_file(result)
    send_notification()
except FileNotFoundError:
    print("文件操作失败")
except ConnectionError:
    print("网络连接失败")
except Exception as e:
    print(f"未知错误：{e}")
```

通过捕获不同类型的异常，可以快速定位问题所在。

### 5. 实际应用场景

- **Web应用**：处理用户表单输入错误
- **文件处理**：处理文件不存在、权限不足等问题
- **数据库操作**：处理连接失败、查询错误等
- **API调用**：处理网络超时、返回数据格式错误等
- **数据转换**：处理类型转换失败、格式不正确等

**总结：** 异常处理不是可选的，而是编写专业、可靠程序的必备技能！

## 基本语法

```python
try:
    # 可能会出错的代码
    代码块
except:
    # 出错后执行的代码
    处理错误的代码
```

## 示例1：处理除零错误

```python
# 不使用异常处理（程序会崩溃）
# num = 10 / 0  # ZeroDivisionError

# 使用异常处理
try:
    num = 10 / 0
except:
    print("错误：不能除以零！")
    num = 0

print(f"结果是：{num}")
```

**输出：**
```
错误：不能除以零！
结果是：0
```

## 示例2：处理用户输入错误

```python
try:
    age = int(input("请输入你的年龄："))
    print(f"你的年龄是：{age}")
except:
    print("输入错误！请输入数字。")
```

**运行示例：**
```
请输入你的年龄：abc
输入错误！请输入数字。
```

## 捕获特定类型的异常

可以针对不同类型的错误进行不同的处理：

```python
try:
    num = int(input("请输入一个数字："))
    result = 100 / num
    print(f"100除以{num}等于{result}")
except ValueError:
    print("错误：你输入的不是数字！")
except ZeroDivisionError:
    print("错误：不能除以零！")
```

## 常见的异常类型

| 异常类型 | 说明 | 示例 |
|---------|------|------|
| `ValueError` | 值错误 | `int("abc")` |
| `ZeroDivisionError` | 除零错误 | `10 / 0` |
| `TypeError` | 类型错误 | `"hello" + 5` |
| `IndexError` | 索引错误 | `list[10]` 当列表只有3个元素 |
| `KeyError` | 键错误 | `dict["不存在的键"]` |
| `FileNotFoundError` | 文件未找到 | 打开不存在的文件 |

## 示例3：处理文件读取错误

```python
try:
    file = open("不存在的文件.txt", "r", encoding="utf-8")
    content = file.read()
    file.close()
except FileNotFoundError:
    print("错误：文件不存在！")
except:
    print("发生了其他错误！")
```

## 使用 else 子句

如果没有发生异常，可以执行 else 中的代码：

```python
try:
    num = int(input("请输入一个数字："))
except ValueError:
    print("输入错误！")
else:
    print(f"你输入的数字是：{num}")
    print("输入成功！")
```

## 使用 finally 子句

finally 中的代码无论是否发生异常都会执行，通常用于清理资源：

```python
try:
    file = open("data.txt", "r", encoding="utf-8")
    content = file.read()
    print(content)
except FileNotFoundError:
    print("文件不存在！")
finally:
    print("程序执行完毕")
    # 如果文件打开了，这里应该关闭它
```

## 完整的异常处理结构

```python
try:
    # 尝试执行的代码
    num = int(input("请输入一个数字："))
    result = 100 / num
except ValueError:
    # 处理值错误
    print("请输入有效的数字！")
except ZeroDivisionError:
    # 处理除零错误
    print("不能除以零！")
except Exception as e:
    # 处理其他所有异常
    print(f"发生错误：{e}")
else:
    # 没有异常时执行
    print(f"计算结果：{result}")
finally:
    # 无论如何都会执行
    print("操作完成")
```

## 示例4：获取异常信息

使用 `as` 关键字可以获取异常的详细信息：

```python
try:
    numbers = [1, 2, 3]
    print(numbers[10])
except IndexError as e:
    print(f"索引错误：{e}")
```

**输出：**
```
索引错误：list index out of range
```

## 实战练习：安全的计算器

```python
def safe_calculator():
    """一个带异常处理的简单计算器"""
    try:
        num1 = float(input("请输入第一个数字："))
        operator = input("请输入运算符（+、-、*、/）：")
        num2 = float(input("请输入第二个数字："))
        
        if operator == "+":
            result = num1 + num2
        elif operator == "-":
            result = num1 - num2
        elif operator == "*":
            result = num1 * num2
        elif operator == "/":
            result = num1 / num2
        else:
            print("无效的运算符！")
            return
            
        print(f"结果：{num1} {operator} {num2} = {result}")
        
    except ValueError:
        print("错误：请输入有效的数字！")
    except ZeroDivisionError:
        print("错误：不能除以零！")
    except Exception as e:
        print(f"发生未知错误：{e}")

# 调用函数
safe_calculator()
```

## 示例5：读取配置文件（带异常处理）

```python
def read_config(filename):
    """安全地读取配置文件"""
    try:
        with open(filename, "r", encoding="utf-8") as file:
            config = file.read()
            return config
    except FileNotFoundError:
        print(f"配置文件 {filename} 不存在，使用默认配置")
        return "默认配置"
    except PermissionError:
        print(f"没有权限读取文件 {filename}")
        return None
    except Exception as e:
        print(f"读取文件时发生错误：{e}")
        return None

# 使用函数
config = read_config("config.txt")
if config:
    print(f"配置内容：{config}")
```

## 最佳实践

1. **具体捕获异常**：尽量捕获特定的异常类型，而不是使用通用的 `except`
2. **不要隐藏错误**：捕获异常后要给出有意义的提示
3. **使用 finally 清理资源**：如关闭文件、数据库连接等
4. **不要过度使用**：只在真正可能出错的地方使用异常处理

## 练习题

1. 编写一个程序，要求用户输入两个数字并计算它们的商，使用异常处理处理可能的错误

2. 编写一个函数，尝试打开并读取一个文件，如果文件不存在则创建它

3. 创建一个列表操作程序，让用户输入索引来访问列表元素，使用异常处理防止索引越界

4. 编写一个字典查询程序，安全地查询字典中的键值，如果键不存在则返回默认值

## 小结

- `try...except` 用于捕获和处理异常
- 可以捕获特定类型的异常进行不同处理
- `else` 在没有异常时执行
- `finally` 无论如何都会执行
- 使用 `as` 可以获取异常详细信息
- 合理使用异常处理可以让程序更加健壮
