# 综合项目：个人图书管理系统（函数版）

## 项目简介

这是一个综合性的Python项目，整合了Python入门课程的所有知识点（不使用类）。你将创建一个完整的个人图书管理系统，包括图书管理、用户管理、借阅记录等功能。

## 知识点覆盖

本项目将使用以下所有知识点：

- ✅ 第1课：变量和数据类型
- ✅ 第2课：数字运算
- ✅ 第3课：字符串基础
- ✅ 第4课：用户输入和输出
- ✅ 第5课：类型转换
- ✅ 第6课：条件语句
- ✅ 第7课：比较和逻辑运算符
- ✅ 第8课：while循环
- ✅ 第9课：for循环
- ✅ 第10课：break和continue
- ✅ 第11课：列表基础
- ✅ 第12课：列表操作
- ✅ 第13课：元组
- ✅ 第14课：字典基础
- ✅ 第15课：字典操作
- ✅ 第16课：函数定义和调用
- ✅ 第17课：函数参数
- ✅ 第18课：返回值
- ✅ 第19课：文件读取
- ✅ 第20课：文件写入
- ✅ 第22课：异常处理

## 项目要求

### 功能要求

1. **图书管理**
   - 添加图书（书名、作者、ISBN、价格、库存数量）
   - 删除图书
   - 修改图书信息
   - 查询图书（按书名、作者、ISBN）
   - 显示所有图书

2. **用户管理**
   - 注册用户（姓名、年龄、会员等级）
   - 用户登录
   - 查看用户信息
   - 显示所有用户

3. **借阅管理**
   - 借书（记录借阅时间、用户信息）
   - 还书（计算借阅天数、费用）
   - 查看借阅记录
   - 统计借阅数据

4. **数据持久化**
   - 将图书数据保存到文件
   - 将用户数据保存到文件
   - 将借阅记录保存到文件
   - 程序启动时自动加载数据

5. **统计功能**
   - 统计图书总数
   - 统计用户总数
   - 统计最受欢迎的图书
   - 计算总收入

### 技术要求

1. 使用**函数**来组织代码
2. 使用**字典**存储图书和用户信息
3. 使用**列表**存储借阅记录
4. 使用**异常处理**确保程序健壮性
5. 使用**文件操作**实现数据持久化
6. 使用**循环和条件语句**实现业务逻辑

## 项目结构

```
library_system.py          # 主程序文件
books.txt                  # 图书数据文件
users.txt                  # 用户数据文件
records.txt                # 借阅记录文件
```

## 数据结构设计

### 全局变量

```python
# 图书数据：字典，键是ISBN，值是图书信息字典
books = {
    "9787115428028": {
        "isbn": "9787115428028",
        "title": "Python编程从入门到实践",
        "author": "Eric Matthes",
        "price": 89.0,
        "stock": 10
    }
}

# 用户数据：字典，键是用户ID，值是用户信息字典
users = {
    "U001": {
        "user_id": "U001",
        "name": "张三",
        "age": 25,
        "level": "VIP会员",
        "borrowed_books": []  # 存储借阅的ISBN列表
    }
}

# 借阅记录：列表，每个元素是一个字典
records = [
    {
        "user_id": "U001",
        "isbn": "9787115428028",
        "borrow_date": "2024-01-01",
        "return_date": "",  # 空字符串表示未归还
        "fee": 0
    }
]

# 当前登录用户
current_user = None  # None表示未登录，否则存储用户ID
```

## 函数设计

### 1. 图书管理函数

```python
def add_book(books):
    """添加图书"""
    try:
        print("\n=== 添加图书 ===")
        isbn = input("请输入ISBN：").strip()
        
        if isbn in books:
            print("错误：该ISBN已存在！")
            return
        
        title = input("请输入书名：").strip()
        author = input("请输入作者：").strip()
        
        try:
            price = float(input("请输入价格："))
            stock = int(input("请输入库存数量："))
        except ValueError:
            print("错误：价格和库存必须是数字！")
            return
        
        if price < 0 or stock < 0:
            print("错误：价格和库存不能为负数！")
            return
        
        # 创建图书字典
        books[isbn] = {
            "isbn": isbn,
            "title": title,
            "author": author,
            "price": price,
            "stock": stock
        }
        
        print(f"成功添加图书：{title}")
        
    except Exception as e:
        print(f"添加图书时发生错误：{e}")


def delete_book(books):
    """删除图书"""
    pass


def search_book(books):
    """查询图书"""
    pass


def display_all_books(books):
    """显示所有图书"""
    if not books:
        print("\n图书库为空！")
        return
    
    print("\n" + "="*80)
    print(f"{'ISBN':<15} {'书名':<30} {'作者':<15} {'价格':<10} {'库存':<10}")
    print("="*80)
    
    for isbn, book in books.items():
        print(f"{book['isbn']:<15} {book['title']:<30} {book['author']:<15} "
              f"¥{book['price']:<9.2f} {book['stock']:<10}")
    
    print("="*80)
    print(f"共有 {len(books)} 本图书")


def update_book(books):
    """修改图书信息"""
    pass
```

### 2. 用户管理函数

```python
def register_user(users):
    """注册用户"""
    try:
        print("\n=== 用户注册 ===")
        
        # 生成用户ID
        user_id = "U" + str(len(users) + 1).zfill(3)
        
        name = input("请输入姓名：").strip()
        
        try:
            age = int(input("请输入年龄："))
        except ValueError:
            print("错误：年龄必须是数字！")
            return None
        
        if age < 0 or age > 150:
            print("错误：年龄不合法！")
            return None
        
        print("会员等级：1-普通会员  2-VIP会员")
        level_choice = input("请选择会员等级（1/2）：").strip()
        level = "VIP会员" if level_choice == "2" else "普通会员"
        
        # 创建用户字典
        users[user_id] = {
            "user_id": user_id,
            "name": name,
            "age": age,
            "level": level,
            "borrowed_books": []
        }
        
        print(f"注册成功！您的用户ID是：{user_id}")
        return user_id
        
    except Exception as e:
        print(f"注册时发生错误：{e}")
        return None


def login(users):
    """用户登录"""
    print("\n=== 用户登录 ===")
    user_id = input("请输入用户ID：").strip()
    
    if user_id in users:
        print(f"登录成功！欢迎您，{users[user_id]['name']}！")
        return user_id
    else:
        print("错误：用户ID不存在！")
        return None


def display_user_info(users, user_id):
    """显示用户信息"""
    if user_id not in users:
        print("错误：用户不存在！")
        return
    
    user = users[user_id]
    print("\n=== 用户信息 ===")
    print(f"用户ID：{user['user_id']}")
    print(f"姓名：{user['name']}")
    print(f"年龄：{user['age']}")
    print(f"会员等级：{user['level']}")
    print(f"已借图书数量：{len(user['borrowed_books'])}")


def display_all_users(users):
    """显示所有用户"""
    pass
```

### 3. 借阅管理函数

```python
def borrow_book(books, users, records, current_user):
    """借书"""
    if current_user is None:
        print("错误：请先登录！")
        return
    
    try:
        print("\n=== 借书 ===")
        isbn = input("请输入要借的图书ISBN：").strip()
        
        # 检查图书是否存在
        if isbn not in books:
            print("错误：图书不存在！")
            return
        
        # 检查库存
        if books[isbn]["stock"] <= 0:
            print("错误：该图书已无库存！")
            return
        
        # 检查用户是否已借此书
        if isbn in users[current_user]["borrowed_books"]:
            print("错误：您已经借阅了这本书！")
            return
        
        # 借阅日期（简化处理，使用输入）
        borrow_date = input("请输入借阅日期（格式：2024-01-01）：").strip()
        
        # 更新库存
        books[isbn]["stock"] -= 1
        
        # 添加到用户借阅列表
        users[current_user]["borrowed_books"].append(isbn)
        
        # 创建借阅记录
        record = {
            "user_id": current_user,
            "isbn": isbn,
            "borrow_date": borrow_date,
            "return_date": "",
            "fee": 0
        }
        records.append(record)
        
        print(f"借阅成功！《{books[isbn]['title']}》")
        
    except Exception as e:
        print(f"借书时发生错误：{e}")


def return_book(books, users, records, current_user):
    """还书"""
    if current_user is None:
        print("错误：请先登录！")
        return
    
    try:
        print("\n=== 还书 ===")
        
        # 显示用户已借图书
        borrowed = users[current_user]["borrowed_books"]
        if not borrowed:
            print("您当前没有借阅任何图书！")
            return
        
        print("您借阅的图书：")
        for i, isbn in enumerate(borrowed, 1):
            print(f"{i}. {books[isbn]['title']} (ISBN: {isbn})")
        
        try:
            choice = int(input("请选择要归还的图书编号："))
            if choice < 1 or choice > len(borrowed):
                print("错误：无效的选择！")
                return
        except ValueError:
            print("错误：请输入数字！")
            return
        
        isbn = borrowed[choice - 1]
        
        # 归还日期
        return_date = input("请输入归还日期（格式：2024-01-15）：").strip()
        
        # 计算费用（简化：每天1元，VIP免费）
        try:
            days = int(input("请输入借阅天数："))
            if users[current_user]["level"] == "VIP会员":
                fee = 0
            else:
                fee = days * 1.0
        except ValueError:
            print("错误：天数必须是数字！")
            return
        
        # 更新库存
        books[isbn]["stock"] += 1
        
        # 从用户借阅列表移除
        users[current_user]["borrowed_books"].remove(isbn)
        
        # 更新借阅记录
        for record in records:
            if (record["user_id"] == current_user and 
                record["isbn"] == isbn and 
                record["return_date"] == ""):
                record["return_date"] = return_date
                record["fee"] = fee
                break
        
        print(f"归还成功！《{books[isbn]['title']}》")
        print(f"借阅费用：¥{fee:.2f}")
        
    except Exception as e:
        print(f"还书时发生错误：{e}")


def display_records(records, books, users):
    """显示借阅记录"""
    if not records:
        print("\n暂无借阅记录！")
        return
    
    print("\n" + "="*100)
    print(f"{'用户':<10} {'书名':<30} {'借阅日期':<12} {'归还日期':<12} {'费用':<10}")
    print("="*100)
    
    for record in records:
        user_name = users[record["user_id"]]["name"]
        book_title = books[record["isbn"]]["title"]
        return_date = record["return_date"] if record["return_date"] else "未归还"
        
        print(f"{user_name:<10} {book_title:<30} {record['borrow_date']:<12} "
              f"{return_date:<12} ¥{record['fee']:<9.2f}")
    
    print("="*100)


def display_my_records(records, books, current_user):
    """显示当前用户的借阅记录"""
    pass
```

### 4. 数据持久化函数

```python
def save_books(books, filename="books.txt"):
    """保存图书数据到文件"""
    try:
        with open(filename, "w", encoding="utf-8") as file:
            # 写入表头
            file.write("ISBN,书名,作者,价格,库存\n")
            
            # 写入每本图书的数据
            for isbn, book in books.items():
                line = f"{book['isbn']},{book['title']},{book['author']},"
                line += f"{book['price']},{book['stock']}\n"
                file.write(line)
        
        print(f"图书数据已保存到 {filename}")
        
    except Exception as e:
        print(f"保存图书数据时发生错误：{e}")


def load_books(filename="books.txt"):
    """从文件加载图书数据"""
    books = {}
    
    try:
        with open(filename, "r", encoding="utf-8") as file:
            lines = file.readlines()
            
            # 跳过表头
            for line in lines[1:]:
                line = line.strip()
                if not line:
                    continue
                
                # 分割数据
                parts = line.split(",")
                if len(parts) == 5:
                    isbn, title, author, price, stock = parts
                    books[isbn] = {
                        "isbn": isbn,
                        "title": title,
                        "author": author,
                        "price": float(price),
                        "stock": int(stock)
                    }
        
        print(f"成功加载 {len(books)} 本图书")
        
    except FileNotFoundError:
        print(f"文件 {filename} 不存在，将创建新文件")
    except Exception as e:
        print(f"加载图书数据时发生错误：{e}")
    
    return books


def save_users(users, filename="users.txt"):
    """保存用户数据到文件"""
    try:
        with open(filename, "w", encoding="utf-8") as file:
            file.write("用户ID,姓名,年龄,会员等级,已借图书\n")
            
            for user_id, user in users.items():
                borrowed = ";".join(user["borrowed_books"])  # 用分号连接
                line = f"{user['user_id']},{user['name']},{user['age']},"
                line += f"{user['level']},{borrowed}\n"
                file.write(line)
        
        print(f"用户数据已保存到 {filename}")
        
    except Exception as e:
        print(f"保存用户数据时发生错误：{e}")


def load_users(filename="users.txt"):
    """从文件加载用户数据"""
    users = {}
    
    try:
        with open(filename, "r", encoding="utf-8") as file:
            lines = file.readlines()
            
            for line in lines[1:]:
                line = line.strip()
                if not line:
                    continue
                
                parts = line.split(",")
                if len(parts) == 5:
                    user_id, name, age, level, borrowed = parts
                    
                    # 处理借阅列表
                    borrowed_books = borrowed.split(";") if borrowed else []
                    borrowed_books = [b for b in borrowed_books if b]  # 过滤空字符串
                    
                    users[user_id] = {
                        "user_id": user_id,
                        "name": name,
                        "age": int(age),
                        "level": level,
                        "borrowed_books": borrowed_books
                    }
        
        print(f"成功加载 {len(users)} 个用户")
        
    except FileNotFoundError:
        print(f"文件 {filename} 不存在，将创建新文件")
    except Exception as e:
        print(f"加载用户数据时发生错误：{e}")
    
    return users


def save_records(records, filename="records.txt"):
    """保存借阅记录到文件"""
    try:
        with open(filename, "w", encoding="utf-8") as file:
            file.write("用户ID,ISBN,借阅日期,归还日期,费用\n")
            
            for record in records:
                line = f"{record['user_id']},{record['isbn']},"
                line += f"{record['borrow_date']},{record['return_date']},"
                line += f"{record['fee']}\n"
                file.write(line)
        
        print(f"借阅记录已保存到 {filename}")
        
    except Exception as e:
        print(f"保存借阅记录时发生错误：{e}")


def load_records(filename="records.txt"):
    """从文件加载借阅记录"""
    records = []
    
    try:
        with open(filename, "r", encoding="utf-8") as file:
            lines = file.readlines()
            
            for line in lines[1:]:
                line = line.strip()
                if not line:
                    continue
                
                parts = line.split(",")
                if len(parts) == 5:
                    user_id, isbn, borrow_date, return_date, fee = parts
                    
                    record = {
                        "user_id": user_id,
                        "isbn": isbn,
                        "borrow_date": borrow_date,
                        "return_date": return_date,
                        "fee": float(fee)
                    }
                    records.append(record)
        
        print(f"成功加载 {len(records)} 条借阅记录")
        
    except FileNotFoundError:
        print(f"文件 {filename} 不存在，将创建新文件")
    except Exception as e:
        print(f"加载借阅记录时发生错误：{e}")
    
    return records


def save_all_data(books, users, records):
    """保存所有数据"""
    print("\n=== 保存数据 ===")
    save_books(books)
    save_users(users)
    save_records(records)
    print("所有数据保存完成！")
```

### 5. 统计函数

```python
def show_statistics(books, users, records):
    """显示统计信息"""
    print("\n" + "="*50)
    print("           系统统计信息")
    print("="*50)
    
    # 图书统计
    total_books = len(books)
    total_stock = sum(book["stock"] for book in books.values())
    
    print(f"图书种类总数：{total_books}")
    print(f"图书库存总数：{total_stock}")
    
    # 用户统计
    total_users = len(users)
    vip_count = sum(1 for user in users.values() if user["level"] == "VIP会员")
    
    print(f"注册用户总数：{total_users}")
    print(f"VIP会员数量：{vip_count}")
    
    # 借阅统计
    total_records = len(records)
    returned = sum(1 for record in records if record["return_date"])
    not_returned = total_records - returned
    
    print(f"总借阅次数：{total_records}")
    print(f"已归还：{returned}")
    print(f"未归还：{not_returned}")
    
    # 收入统计
    total_income = sum(record["fee"] for record in records)
    print(f"总收入：¥{total_income:.2f}")
    
    # 最受欢迎的图书
    if records:
        book_count = {}
        for record in records:
            isbn = record["isbn"]
            book_count[isbn] = book_count.get(isbn, 0) + 1
        
        if book_count:
            most_popular_isbn = max(book_count, key=book_count.get)
            most_popular_title = books[most_popular_isbn]["title"]
            print(f"最受欢迎图书：《{most_popular_title}》({book_count[most_popular_isbn]}次)")
    
    print("="*50)
```

### 6. 菜单和主程序

```python
def display_menu(current_user, users):
    """显示主菜单"""
    print("\n" + "="*50)
    print("           个人图书管理系统 v1.0")
    print("="*50)
    
    if current_user:
        print(f"当前用户：{users[current_user]['name']} ({current_user})")
    else:
        print("当前用户：未登录")
    
    print("\n【用户管理】")
    print("1. 注册新用户")
    print("2. 用户登录")
    print("3. 查看个人信息")
    
    print("\n【图书管理】")
    print("4. 添加图书")
    print("5. 删除图书")
    print("6. 修改图书信息")
    print("7. 查询图书")
    print("8. 显示所有图书")
    
    print("\n【借阅管理】")
    print("9. 借书")
    print("10. 还书")
    print("11. 查看借阅记录")
    
    print("\n【系统功能】")
    print("12. 数据统计")
    print("13. 保存数据")
    print("0. 退出系统")
    print("="*50)


def main():
    """主程序"""
    print("欢迎使用个人图书管理系统！")
    print("正在加载数据...")
    
    # 加载数据
    books = load_books()
    users = load_users()
    records = load_records()
    
    # 当前登录用户
    current_user = None
    
    # 主循环
    while True:
        display_menu(current_user, users)
        
        try:
            choice = input("\n请选择功能（0-13）：").strip()
            
            if choice == "0":
                # 退出前保存数据
                save_choice = input("是否保存数据？(y/n)：").strip().lower()
                if save_choice == "y":
                    save_all_data(books, users, records)
                print("感谢使用，再见！")
                break
            
            elif choice == "1":
                register_user(users)
            
            elif choice == "2":
                current_user = login(users)
            
            elif choice == "3":
                if current_user:
                    display_user_info(users, current_user)
                else:
                    print("请先登录！")
            
            elif choice == "4":
                add_book(books)
            
            elif choice == "5":
                delete_book(books)
            
            elif choice == "6":
                update_book(books)
            
            elif choice == "7":
                search_book(books)
            
            elif choice == "8":
                display_all_books(books)
            
            elif choice == "9":
                borrow_book(books, users, records, current_user)
            
            elif choice == "10":
                return_book(books, users, records, current_user)
            
            elif choice == "11":
                display_records(records, books, users)
            
            elif choice == "12":
                show_statistics(books, users, records)
            
            elif choice == "13":
                save_all_data(books, users, records)
            
            else:
                print("无效的选择，请重新输入！")
        
        except Exception as e:
            print(f"发生错误：{e}")
            print("请重试！")


# 程序入口
if __name__ == "__main__":
    main()
```

## 实现步骤

### 第一阶段：数据结构设计（使用第1-5课、第11-15课知识）

1. 设计图书字典结构
2. 设计用户字典结构
3. 设计借阅记录列表结构
4. 理解数据类型和类型转换

### 第二阶段：基础函数（使用第16-18课知识）

1. 实现添加图书函数
2. 实现注册用户函数
3. 实现显示函数
4. 理解函数参数和返回值

### 第三阶段：业务逻辑（使用第6-10课知识）

1. 实现菜单系统（while循环）
2. 实现条件判断（用户权限、库存检查）
3. 实现数据遍历（for循环显示列表）
4. 使用break和continue控制流程

### 第四阶段：数据持久化（使用第19-20课知识）

1. 实现保存图书到文件
2. 实现从文件加载图书
3. 实现保存用户和记录
4. 处理文件读写异常

### 第五阶段：异常处理（使用第22课知识）

1. 处理用户输入错误
2. 处理文件操作异常
3. 处理数据转换异常
4. 确保程序稳定运行

### 第六阶段：完善功能

1. 实现删除、修改、查询功能
2. 实现统计功能
3. 优化用户体验
4. 测试所有功能

## 待实现的函数


```python
def delete_book(books):
    """删除图书 - 待实现"""
    # 提示：
    # 1. 输入ISBN
    # 2. 检查图书是否存在
    # 3. 使用del删除字典中的键值对
    pass

def search_book(books):
    """查询图书 - 待实现"""
    # 提示：
    # 1. 提供多种查询方式（按书名、作者、ISBN）
    # 2. 使用for循环遍历字典
    # 3. 使用in运算符进行模糊搜索
    pass

def update_book(books):
    """修改图书信息 - 待实现"""
    # 提示：
    # 1. 输入ISBN找到图书
    # 2. 显示当前信息
    # 3. 输入新信息并更新字典
    pass

def display_all_users(users):
    """显示所有用户 - 待实现"""
    # 提示：
    # 1. 检查用户字典是否为空
    # 2. 使用for循环遍历并打印
    # 3. 格式化输出
    pass

def display_my_records(records, books, current_user):
    """显示当前用户的借阅记录 - 待实现"""
    # 提示：
    # 1. 过滤出当前用户的记录
    # 2. 格式化显示
    pass
```

## 评分标准

### 基础功能（60分）
- [ ] 数据结构设计正确（10分）
- [ ] 添加、删除、查询图书（15分）
- [ ] 用户注册和登录（15分）
- [ ] 菜单系统和主循环（10分）
- [ ] 基本的输入输出（10分）

### 核心功能（30分）
- [ ] 借书功能（10分）
- [ ] 还书功能（10分）
- [ ] 数据持久化（10分）

### 高级功能（10分）
- [ ] 异常处理（5分）
- [ ] 统计功能（3分）
- [ ] 代码规范和注释（2分）

## 提示和建议

### 关键技术点

1. **字典操作**
   - 添加：`books[isbn] = {...}`
   - 删除：`del books[isbn]`
   - 检查：`if isbn in books:`
   - 遍历：`for isbn, book in books.items():`

2. **列表操作**
   - 添加：`records.append(record)`
   - 删除：`records.remove(record)`
   - 遍历：`for record in records:`

3. **文件操作**
   - 写入：`file.write(line)`
   - 读取：`lines = file.readlines()`
   - 分割：`parts = line.split(",")`
   - 连接：`",".join(list)`

4. **异常处理**
   ```python
   try:
       # 可能出错的代码
   except ValueError:
       # 处理值错误
   except FileNotFoundError:
       # 处理文件不存在
   except Exception as e:
       # 处理其他错误
   ```

### 测试建议

1. 先实现基本功能，确保能运行
2. 逐个测试每个函数
3. 测试异常情况（错误输入、文件不存在等）
4. 测试数据保存和加载
5. 进行完整的业务流程测试

## 完成标准

完成以下所有功能即可视为项目完成：

1. ✅ 能够添加、删除、查询、修改图书
2. ✅ 能够注册用户和登录
3. ✅ 能够借书和还书
4. ✅ 能够保存和加载数据
5. ✅ 程序运行稳定，有完善的异常处理
6. ✅ 代码结构清晰，有适当的注释

---
